#!/bin/bash

SIZES="4 6 8 10 12 14 16 18 20"

DOC="
  Test conjecture that every bridgeless cubic graph has
  at least 2^{n/2 - 1} circuit double covers on all
  graphs of given size. We know that minimal
  counterexample does not have 3 or 4-cycle so we skip
  those.

  The arguments should be the sizes of the graphs
  to be tested. If it is empty it defaults to
  $SIZES.

  It requires geng from Nauty package to be in the PATH.

  WARNING: This experiment takes a long time to complete.
"

echo "$DOC"

export PYTHONUNBUFFERED=x

if [[ -n "$*" ]]; then
  SIZES="$*"
fi

: ${LOG:=${0%.sh}-$SIZES.log}
: ${GENG:=geng}
: ${PYTHON:=python}

SELF="$(readlink -f "$0")"

for s in $SIZES; do
  "$GENG" -Ctfd3D3 $s
done |
nice $PYTHON /dev/fd/5 "$(dirname "$SELF")" "$LOG" 5<<'EOF'
if True:
  import sys, os
  sys.path.append(sys.argv[1] + "/..")

  import json
  from graph_tools.misc import graph_to_gadget
  from graph_tools.parameters import CircuitDoubleCover, UnderlyingGraph
  from sage.all import Graph
  from parmap import parmap

  def process(l):
    G = Graph(l)
    gadget = graph_to_gadget(G)
    real = gadget.eval(CircuitDoubleCover)
    exp = 2**(G.num_verts() // 2) // 2
    assert G.is_isomorphic(gadget.eval(UnderlyingGraph))
  
    j = {
      "graph": G.sparse6_string(),
      "cdc_count": real,
      "n": G.num_verts(),
      "expected_cdc_lowerbound": exp,
      "is_ok": (real >= exp)
    }

    return json.dumps(j), j["is_ok"]

  inp = os.fdopen(sys.stdin.fileno())
  sys.stdin = None

  nprocs = int(os.getenv("NPROCS", "0")) or None
  with open(sys.argv[2], "w") as log:
    sys.stdout.write(" Processed 0   (failed 0)")
    failed = 0
    for i, (l, is_ok) in enumerate(parmap(process, inp, nprocs=nprocs, in_order=False)):
      log.write(l + "\n")
      failed += not is_ok
      sys.stdout.write("\033[1K\r Processed %i   (failed %i)" % (i+1, failed))
      sys.stdout.flush()

  print("\nDone")
EOF

